% *************************************************************************************************************************************
% 
% Program :  Computes the aggreagated and full infinite and finite population models
% Author  :  S.Y. Yuen   (with help from Q. Li)
% Version :  16th March 2005
%
% Full Paper :  Bounds for Probability of Success of Classical Genetic Algorithm based on Hamming Distance

% INPUT PARAMETERS
% *************************************************************************************************************************************

% reset
clear;clc;close all;

disp ('GA Simulator');
disp (' ');

disp ('Note: ');
disp ('1. This program may allow zero fitness value, but will fail if the fitness landscape is a plane of zero fitness');
disp ('2. All fitness values must be non-negative');
disp ('3. Selecting choice 1 will run much quicker since it works solely on the aggregated infinite population');
disp ('4. Please transform manually your fitness function to the form where (1 ... 1) is the optimal solution first');
disp ('5. The program can handle special cases like mutation rate and/or crossover rate = 0');
disp ('6. Sometimes, MATLAB can be stopped by typing ctrl-c');
disp ('7. Code optimization:  The permutation indices of aggregated finite population model is pre-stored');
disp ('   If you have not stored the permutation indices for the (gamma, poolsize) combination, you need to');
disp ('   run    Genperindex.m first');
disp ('8. Code optimization not done: pre-store aggregated infinite population model mixing matrix');
disp (' ');

disp ('choice');
disp ('1           -  aggregated infinite population models only');
disp ('2           -  (aggregated and full) infinite population models only');
disp ('3           -  2. and approximate aggregated finite population model only');
disp ('4           -  3. and the lower and upper bound aggregated finite population models');
disp ('5           -  both infinite and finite population models');
disp (' ');
choice =    input  ('choice                                      = ');

disp (' ');

if choice < 1 | choice > 5
   disp ('FATAL ERROR !  Invalid choice.');
end



% input the variables
disp ('EA parameters');
chi_mu    = input ('bit crossover rate (uniform crossover)      = ');
mu        = input ('bit mutation  rate (uniform mutation )      = ');

no_of_gen = input ('no. of generations (>=1, gen. 0 is the initial generation) = ');

gamma     = input ('chromosome length                           = ');

if choice == 3 | choice == 4 | choice == 5
   poolsize  = input ('population pool size                        = ');
end

disp (' ');

disp ('fitness function');
disp ('1         -  counting one          ');
disp ('2         -  linear (or modular)   ');
disp ('3         -  pseudo modular        ');
disp ('4         -  almost positive       ');
disp ('5         -  real royal road       ');
disp ('6         -  input landscape       ');

functype=input       ('choose fitness function number (1-6)        = ');
disp (' ');

if functype == 2 % If choose the set of funciton 2, choose the sub-function of type 2 for different Ci
   disp ('1         -   ci = i     for function 2');
   disp ('2         -   ci = 2^i                 ');
   disp ('3         -   ci = 2^(-i)              ');
   
   functype2=input('choice            = ');  

   if functype2 < 1 | functype2 > 3
   disp ('FATAL ERROR !  Invalid choice.');
   
   disp (' ');
end

   
end
randamp =      input ('maximum of added uniform random noise (>= 0, 0 if no random noise)    = ');

disp (' ');

colour_plot = input ('colour plot (0 : black and white, 1 : colour)   = ');

disp (' ');


tic

% FITNESS LANDSCAPE AND FITNESS UPPER AND LOWER BOUNDS 
% *************************************************************************************************************************************

W             = 2^gamma;      % search space size
optimal_index = W;            % the optimal index is the integer index of the binary vector of (1 ... 1). Here 1 means a correct bit
                              % and 0 means an incorrect bit

if ~(choice ==1)                              
             
    % ** assign fitness **
    Fitness=ones(1,W); %  Fitness function vector


    binary_pattern=[];   % binary_pattern is [0 0 0 0 ...]    It stores all binary patterns column by column
                         %                   [0 0 1 1 ...]
                         %                   [0 1 0 1 ...]

    % generate fitness vector and aggregated all individuals to ring
    if functype~=6
        for ii=1:W
            if ii/1024 == round(ii/1024)
                disp (ii)
            end
            X=dec2bin(ii-1,gamma);           % convert integer ii-1 to binary 
            X=str2num(X(:));                 % convert binary to column vector
            binary_pattern=[binary_pattern X];               % append it to binary
            
            % generate the fitness value for integer ii-1
            if(functype ==2) 
                Fitness(ii) =generateF2(X,gamma,functype2) ; 
            else     
                Fitness(ii) =generateF (X,gamma,functype) ;
            end    
        end
    else
        fid = fopen('cvtlandscape', 'r');
        for ii=1:W
            if ii/1024 == round(ii/1024)
                disp (ii)
            end
            X=dec2bin(ii-1,gamma);           % convert integer ii-1 to binary 
            X=str2num(X(:));                 % convert binary to column vector
            binary_pattern=[binary_pattern X];               % append it to binary
            
            % load fitness value from file
            Fitness(ii) = fscanf(fid, '%f', 1); 
        end
        fclose(fid);
    end

    % ** add random noise
    if randamp > 0
       
       Noise = randamp * rand (1, W);
       
       Noise_mse = sum(Noise.*Noise)/W;
       
       Signal_var = var(Fitness);
       
       disp ('Random numbers added to everywhere except the global optimum');
       disp (' ');
       
       disp (['Variance of Fitness        = ', num2str(Signal_var)]);
       disp (['mean square error of noise = ', num2str(Noise_mse) ]);
       disp (['SNR                        = ', num2str(10 * log10(Signal_var/Noise_mse)), 'dB' ]);
       disp (' ');
             
       temp    = Fitness(optimal_index);   % no noise is added to the global optima
       Fitness = Fitness + Noise;
       Fitness(optimal_index) = temp;
       
       if max(Fitness) ~= Fitness(optimal_index)
          disp ('Fatal error : too much noise has been added. The location of global optimum has been changed!');
          input ('press any key to continue; ctrl-c to exit ');
       end
    end
end


% aggregated infinite population model
upbound  = ones (1, gamma+1);  % fitness upper bound of each ring
lowbound = ones (1, gamma+1);  % fitness lower bound of each ring

% generate fitness bound of each ring

% ** calculate lower and upper bound analytically **
if functype~=6
    for ii=1:gamma+1
        
        switch functype
            case 1
                lowbound(ii)=(ii-1);     % fitness = 0 for ring 0
                upbound (ii)=(ii-1);
            case 2
                switch functype2
                    case 1
                        lowbound(ii)=(ii-1)*(ii)/2;                  % fitness = 0 for ring 0
                        upbound (ii)=(ii-1)*(2*gamma-(ii-1)+1)/2;        
                    case 2
                        lowbound(ii)=2*(2^(ii-1)-1)+1;           
                        upbound(ii) =2^(gamma-(ii-1)+1)*(2^(ii-1)-1)+1;
                    case 3
                        lowbound(ii)=2^(-(gamma-(ii-1)))*(1-2^-(ii-1))+1; 
                        upbound(ii) =1-2^(-(ii-1))+1;
                end
            case 3
                if(ii~=gamma+1)
                    lowbound(ii)=0;             % fitness can be 0
                    upbound(ii)=ii-1;
                else
                    lowbound(ii)=gamma;
                    upbound(ii)=gamma;
                end
            case 4
                if(ii~=gamma+1)
                    lowbound(ii)=gamma-ii+1;
                else
                    lowbound(ii)=gamma+1;
                end
                upbound(ii)= lowbound(ii);
            case 5
                tao=round(gamma/2);
                if(ii-1<=gamma-tao) | ii-1==gamma
                    lowbound(ii)=tao+ii-1;
                else
                    lowbound(ii)=gamma-(ii-1);
                end
                upbound(ii)= lowbound(ii);
                
        end
    end     
else
    fid = fopen('LUBound', 'r');
    for ii = 1: gamma + 1
        lowbound(ii) = fscanf(fid, '%f', 1);
        upbound(ii) = fscanf(fid, '%f', 1);
    end
    fclose(fid);
end
   
% add the effect of noise to the bounds.  Note that only the upper bound will be affected. The upper bound of the optimal index is 
% unchanged
upbound(1:gamma) = upbound(1:gamma) + randamp;
   
if max(upbound(1:gamma)) > upbound(gamma+1)
   disp ('Warning : upper bound of non-optimal ring is larger than the optimal value!'); 
end
       
       
if choice ~= 1
    % display the fitness function
    figure; hold on;
    if functype ~=2
       title (['fitness function ', num2str(functype) ]);
    else
       switch functype2
           case 1
               title (['fitness function 2 ci = i']);
           case 2
               title (['fitness function 2 ci = 2^i']);
           case 3
               title (['fitness function 2 ci = 2^-^i']);
       end
    end
        
    x=0:W-1;
    plot(x, Fitness, 'k-');

    hold off
end

if choice ~= 1
   Fitness
end

lowbound

upbound

disp (['time to compute fitness and bounds = ', num2str(toc) ]);


% INFINITE POPULATION MODEL 
% *************************************************************************************************************************************

if choice ~= 1     
    
% FULL MODEL (infinite population model)
% ***********************************************************************************************

tic

% generate full model mixing matrix M
M = generateM(mu, chi_mu, gamma);

disp (['time to compute mixing matrix M = ', num2str(toc) ]);

tic

expgamma=gamma-1:-1:0; 
expgamma=2.^expgamma;             % eg. for gamma=3,  expgamma=[4,2,1] 
expgamma=expgamma'*ones(1,W);     % expgamma is a data structure for the XOR operation in 
                                  % Vose and Liepins' "Punctuated Equilibria in Genetic Search" pg. 33
                                  % expgamma = [4 4 4 ... 4]
                                  %            [2 2 2 ... 2]
                                  %            [1 1 1 ... 1]

% initialize probability vector
S=ones(1,W); %  probability vector for infinite population model
S=S/W;       %  equally distributed
MS=ones(1,W);%  S after selction, crossover and mutation

% initialize the expected probability of optimal chromosome in infinite population modle
infi_EP(1)    =S(optimal_index); % Note: index 1 means the initial generation, or generation 0

for iter=1:no_of_gen
    % disp (['full infinite model: generation ', num2str(iter)]);
    
    S=Fitness.*S;
    S=S/sum(S);
     for index=1:W
         delta= binary_pattern(:,index);     % convert the index to binary 
         delta = delta(:)*ones(1,W);
         % to do delta operation as in pg. 33 of "Punctuated Equilibria in Genetic Search" 
         delta=xor(delta,binary_pattern);
         delta=delta.*expgamma;      % these 2 lines change it to integer index
         delta=sum(delta);           %
         MS(index)=S(delta+1)*M*S(delta+1)';
     end
     S=MS;
     infi_EP(length(infi_EP)+1)=S(optimal_index);
     
end

disp (['time to compute the full infinite population model = ', num2str(toc) ]);

disp (' ');
disp ('full Infinite Population Model finished');
disp (' ');

end  % choice


% AGGREGATED MODEL (infinite population model)
% ***********************************************************************************************

tic

% generate mixing matrix Agg_M based on independent equally liklihood model
Agg_M=Agg_getmatrixM(gamma,mu,chi_mu);

disp (['time to compute aggregated mixing matrix = ', num2str(toc) ]);

tic

% initialize probability
R_p_ini    =zeros(1,gamma+1); % probability vector of ring in infinite population model
R_lowbound =zeros(1,gamma+1); % lower bound probability vector of ring in infinite population model
R_upbound  =zeros(1,gamma+1); % upper bound probability vector of ring in infintie population model

ring_num = [];
for i = 1:gamma+1
    ring_num = [ring_num nchoosek(gamma, i-1)];
end

R_p_ini    = ring_num/W;      % equally distributed
R_lowbound = R_p_ini;
R_upbound  = R_p_ini;

% initialize the expected probability of optimal chromosome in infinite population model
infi_low_EP(1)=R_p_ini(gamma+1); % Note: index 1 means the initial generation, or generation 0
infi_up_EP(1) =R_p_ini(gamma+1);


% these two parameters are to compute the normalization factor of each generation of lower
% bound and upper  bounds to prevent numerical instability
factor1=1;
factor2=1;

for iter=1:no_of_gen
    % disp (['aggregated infinite model: generation ', num2str(iter)]);
    
    R_lowbound=lowbound.*R_lowbound;
    R_upbound =upbound.*R_upbound;

    factor1= factor1*( sum(R_lowbound)/sum(R_upbound) )^2;
    factor2= factor2*( sum(R_upbound) /sum(R_lowbound))^2;
    
    R_lowbound = R_lowbound/sum(R_lowbound);
    R_upbound = R_upbound/sum(R_upbound);

    p_all=reshape((R_lowbound'*R_lowbound)',(gamma+1)^2,1);
    R_lowbound=(Agg_M*p_all)';

    p_all=reshape((R_upbound'*R_upbound)', (gamma+1)^2, 1);
    R_upbound =(Agg_M*p_all)';
        
    infi_low_EP(length(infi_low_EP)+1)= R_lowbound(gamma+1) /(R_lowbound(gamma+1)+factor2*sum(R_upbound(1:gamma)));
    infi_up_EP (length(infi_up_EP) +1)= R_upbound(gamma+1)  /(R_upbound (gamma+1)+factor1*sum(R_lowbound(1:gamma)));  
   
end

disp (['time to compute aggregated infinite population model = ', num2str(toc) ]); 

disp (' ');
disp ('aggregated Infinite Population Model finished');
disp (' ');

% DISPLAY - INFINITE POPULATION MODEL
% ***********************************************************************************************

% plot the expected probability of optimal chromosome
figure; hold on;

if choice ~= 1 
   title ('Expected prob. of optimal chromosome in infinite population model');
   xlabel('generation'); ylabel('probability');

   x = 0:no_of_gen;
   if colour_plot
        plot(x, infi_up_EP, 'r-');
        plot(x, infi_EP,    'g-');
        plot(x, infi_low_EP,'b-');   
   else
        plot(x, infi_up_EP, 'k:');
        plot(x, infi_EP,    'k-');
        plot(x, infi_low_EP,'k--');
   end
   
   legend('upper bound', 'actual', 'lower bound', 0);
   
   max_up_error  = 0;
   max_low_error = 0;

   for x = 1:no_of_gen+1
       temp = abs (infi_up_EP(x) - infi_EP(x));
       if temp > max_up_error
          max_up_error = temp;    
       end
       temp = abs (infi_EP(x) - infi_low_EP(x));
       if temp > max_low_error
          max_low_error = temp;
       end
   end
   
   disp (['maximum upper bound error = ', num2str(max_up_error) ]);
   disp (['maximum lower bound error = ', num2str(max_low_error)]);
   
else
   title ('Expected prob. of optimal chromosome in infinite population model');
   xlabel('generation'); ylabel('probability');

   x = 0:no_of_gen;
   if colour_plot
      plot(x, infi_up_EP, 'r-');
      plot(x, infi_low_EP,'b-');
   else
      plot(x, infi_up_EP, 'k:');
      plot(x, infi_low_EP,'k--');
   end    
   
   legend('upper bound', 'lower bound', 0);    
    
end
   
hold off



% FINITE POPULATION MODEL
% *************************************************************************************************************************************

if choice == 3 | choice == 4  | choice == 5

disp (' ');
disp ('finite population model calculation starts');
disp (' ');    
    

    
% FULL MODEL (finite population model)
% ***********************************************************************************************

if choice == 4  | choice == 5 

tic

Z = generateZ(poolsize,W);    % generate (poolsize+W-1, poolsize) x W matrix of all possible finite populations

%total population size
totalpopu=size(Z,1);

% Q matrix 
Q=zeros(size(Z,1));

% calculate Q
for x=1:totalpopu          % calculating Q(x -> y)
    %x
    
    S=Z(x,:);              % row vector
         
    % division of zero problem taken care of here                        
                           
    if sum(Fitness.*S) ~= 0
       S = Fitness.*S;
    end;                   % if all chromosomes have zero fitness, then let all of them have equal fitness    ZERO fitnes
    
    S = S./sum(S);      
        
    % use the infinite population model
    for index=1:W
        delta= binary_pattern(:,index);        % convert the index to binary
        delta = delta(:)*ones(1,W);
        % to do delta operation as in pg. 33 of "Punctuated equlibria in Genetic Search" 
        delta=xor(delta,binary_pattern);
        delta=delta.*expgamma;         % these 2 lines change it to integer index
        delta=sum(delta);
        MS(index)=S(delta+1)*M*S(delta+1)';
    end            
    S=MS; 
          
    for y=1:totalpopu
        % to calculate Q
        % {Si}.^Zi,y
         
        fact=find(Z(y,:)~=0);
        f=1;
        for l=1:length(fact)
            f=f*factorial(Z(y,fact(l)));
        end 
               
        temps=S(fact).^Z(y,fact);
        temps=prod(temps);

        Q(x,y)=(factorial(poolsize)/f)*temps;  
    end % end of y for loop
    
end % end of x for loop

% generate T matrix
                                  
tempZ=Z(:,optimal_index);         % get the optimal column  
    
[indT]=find(tempZ==0);
[opt_ind] = find(tempZ~=0);

T=Q(indT(:),indT(:));             % the submatrix which involves transition from population x to population y
                                  % of which both populations do not contain the optimal chromosome

disp (['time to compute Q matrix = ', num2str(toc) ]);


end % if choice == 4 | choice == 5

if choice == 5

tic

% initialize the finite population probability vector
popup=zeros(size(Z,1),1);           % column vector 
for i=1:size(Z,1)                   % for all possible finite population
    fact=find(Z(i,:)~=0);
    f=1;
    for l=1:length(fact)
        f=f*factorial(Z(i,fact(l)));  
    end
    popup(i) = (factorial(poolsize)/f)*((1.0/W).^poolsize);
end
                            
popupT=popup(indT(:));            % initial probability of populations which do not contain the optimal chromosome
    
I=eye(size(T));                   % an identity matrix the same size as T

T_first=(sum(inv(I-T),2))'*popupT;  % expected time to find the optimum

% calculate lower probability bound by applying Markov's inequality
P_Markov = [0];
for ii=1:no_of_gen
    if 1 - T_first/ii > 0
       P_Markov = [P_Markov 1 - T_first/ii];
    else
       P_Markov = [P_Markov 0];
    end
end

tempT = I;
P_first(1) = 1 - (1-1/W)^poolsize; 
for ii=1:no_of_gen
    tempT = T * tempT;
    P_first(length(P_first)+1)= 1-(sum(tempT,2))'*popupT;
end

fini_EP(1) = 1 - sum(popupT);
temp = popup;
for ii = 1:no_of_gen
    temp = Q' * temp;
    fini_EP(length(fini_EP)+1) = sum( temp(opt_ind) ); 
end

disp (['time to compute the full finite population model = ', num2str(toc)]);

disp (' ');
disp ('full finite population model finished');
disp (' ');

end % choice == 5

% AGGREGATED MODEL (finite population model)
% ***********************************************************************************************

% Note : Code optimization has been done here.  The permutation indices array label has been pre-computed using
%        function Genperindex

tic

Z = generateZ(poolsize,W);    % generate combination(poolsize+W-1, poolsize) x W matrix of all possible finite populations

label_co     = zeros (size(Z, 1), 1);
label_co     = csvread (['label', num2str(gamma), '_', num2str(poolsize)]);

curr_label = 1;
label     = zeros (size(Z, 1), 1);

for i=1:size(Z, 1)
    if label(i) == 0
       label(i)   = curr_label;
       labelp(curr_label).size   =1;
       labelp(curr_label).pattern=Z(i,:);
          
       for j=i+1:size(Z, 1)
           if label_co(j) == curr_label
              label(j) = curr_label;
              labelp(curr_label).size = labelp(curr_label).size+1;
           end
       end
       
       curr_label = curr_label+1;
    end
end

no_label = curr_label-1;

disp (['time to compute the permutation indices (after using pre-computed file) = ', num2str(toc)]);


tic

% construct the aggregated matrix
Agg_Q = zeros (no_label);


for x = 1:no_label

    for i = 1:size(Z,1)
        
        if label(i) == x
           S=Z(i,:);              % row vector
         
           % division of zero problem taken care of here
                           
           if sum(Fitness.*S) ~= 0
              S = Fitness.*S;
           end;                   % if all chromosomes have zero fitness, then let all of them have equal fitness    ZERO fitnes
    
           S = S./sum(S);      
        
           % use the infinite population model
           for index=1:W
               delta= binary_pattern(:,index);        % convert the index to binary
               delta = delta(:)*ones(1,W);
               % to do delta operation as in pg. 33 of "Punctuated equlibria in Genetic Search" 
               delta=xor(delta,binary_pattern);
               delta=delta.*expgamma;         % these 2 lines change it to integer index
               delta=sum(delta);
               MS(index)=S(delta+1)*M*S(delta+1)';
           end            
           S=MS;  
            
           for y=1:no_label
               temp = find ( label== y );
               % ** implement Agg_Q (x, y) = sum ( Q(i, temp) );
      
               Agg_Q (x, y) = 0;
               for j = 1:length(temp)
        
                   fact=find(Z(temp(j),:)~=0);
                   f=1;
                   for l=1:length(fact)
                       f=f*factorial(Z(temp(j),fact(l)));
                   end 
               
                   temps=S(fact).^Z(temp(j),fact);
                   temps=prod(temps);

                   Agg_Q (x, y) = Agg_Q (x, y) + (factorial(poolsize)/f)*temps;  
               end
               % **
                  
           end
        
           break;
        end  % if label(i) == x
        
    end
    
end


disp (['time to compute the aggregated Q matrix = ', num2str(toc)]);


if choice == 4 | choice == 5

tic    
    
low_Q = zeros (no_label);
up_Q  = zeros (no_label);
    
for x = 1:no_label
    for y=1:no_label
        
        b = [ ];
        for i=1:size(Z, 1)
            if label(i) == x
               temp = find ( label== y );
               b = [b sum(Q(i, temp))];
            end  
        end
        low_Q (x, y) = min (b);
        up_Q  (x, y) = max (b); 
    end
end

disp (['time to compute the aggregated lower and upper bound Q matrix = ', num2str(toc)]);

end % choice == 4  | choice == 5

tic

% initialize the aggregated finite population probability vector
Agg_popupZ = zeros (no_label, 1);           % column vector
for curr_label=1:no_label
    fact=find( labelp(curr_label).pattern );
    f=1;
    for l=1:length(fact)
        f=f*factorial( labelp(curr_label).pattern(fact(l)) );
    end
    Agg_popupZ(curr_label)=labelp(curr_label).size* (factorial(poolsize)/f)*((1.0/W).^poolsize);
end

no_Tlabel = no_label;

tempZ = zeros (no_label, 1);
for curr_label=1:no_label
    fact = labelp(curr_label).pattern;

    if fact(W) > 0    % at least one optimal chromosome in the pattern
       tempZ (curr_label) = 1;
       no_Tlabel = no_Tlabel - 1;
    end
end

Agg_T = zeros (no_Tlabel);

[indT]=find(tempZ == 0);
[opt_ind] = find(tempZ ~= 0);

Agg_T = Agg_Q (indT(:), indT(:));

if  choice == 4 | choice == 5
    low_T = zeros (no_Tlabel);
    up_T  = zeros (no_Tlabel);

    low_T = low_Q (indT(:), indT(:));
    up_T  = up_Q  (indT(:), indT(:));

end % choice == 4 | choice == 5

Agg_popupT = Agg_popupZ (indT);

temp = find ( sum (Agg_T, 2) > 1);

if temp ~= [ ] 
   disp ('approx. T matrix is not a stochastic matrix.  This should not happen!');
end

if  choice == 4 | choice == 5
    temp = find ( sum (low_T, 2) > 1);
    
    if  temp ~= [ ] 
        disp ('lower bound T matrix is not a stochastic matrix.  This should not happen!');
        low_valid = 0;
    else
        low_valid = 1;
    end

    temp = find ( sum (up_T, 2) > 1);
    
    if  temp ~= [ ] 
        disp ('upper bound T matrix is not a stochastic matrix');
        disp (' ');
        up_valid = 0;
    else
        up_valid = 1;
    end
end % choice == 4 | choice == 5


I=eye(size(Agg_T));                       % an identity matrix the same size as Agg_T, low_T and up_T

if choice == 4 | choice == 5
   low_T_first=(sum(inv(I-low_T),2))'* Agg_popupT;  % lower bound of the expected time to find the optimum
   up_T_first =(sum(inv(I-up_T ),2))'* Agg_popupT;  % upper bound of the expected time to find the optimum
end % choice == 4 | choice == 5

Agg_T_first =(sum(inv(I-Agg_T ),2))'* Agg_popupT;  % approx. expected time to find the optimum



% calculate lower, actual and upper probability bounds by applying Markov's inequality

Agg_P_Markov = [0];
for ii=1:no_of_gen
    if 1 - Agg_T_first/ii > 0
       Agg_P_Markov = [Agg_P_Markov 1-Agg_T_first/ii];
    else
       Agg_P_Markov = [Agg_P_Markov 0];
    end
end

if  choice == 4 | choice == 5
    low_P_Markov = [0];
    for ii=1:no_of_gen
        if  1 - low_T_first/ii > 0
            low_P_Markov = [low_P_Markov 1-low_T_first/ii];
        else
            low_P_Markov = [low_P_Markov 0];
        end
    end

    up_P_Markov = [0];
    for ii=1:no_of_gen
        if  1 - up_T_first/ii > 0
            up_P_Markov = [up_P_Markov 1-up_T_first/ii];
        else
            up_P_Markov = [up_P_Markov 0];
        end
    end

end % choice == 4 | choice == 5

% calculate lower, approximate, and upper first passage probability by first passage MC theory

Agg_P_first(1) = 1 - (1-1/W)^poolsize;        
tempT = eye(size(Agg_T));
for ii=1:no_of_gen
    tempT = Agg_T * tempT;
    Agg_P_first(length(Agg_P_first)+1)= 1-(sum(tempT,2))'*Agg_popupT;
end

if  choice == 4 | choice == 5
    low_P_first(1) = 1 - (1-1/W)^poolsize;  
    tempT = eye(size(up_T));
    for ii=1:no_of_gen
        tempT = up_T * tempT;
        low_P_first(length(low_P_first)+1)= 1-(sum(tempT,2))'*Agg_popupT;
    end

    up_P_first(1) = 1 - (1-1/W)^poolsize;        
    tempT = eye(size(low_T));
    for ii=1:no_of_gen
        tempT = low_T * tempT;
        up_P_first(length(up_P_first)+1)= 1-(sum(tempT,2))'*Agg_popupT;
    end
end % choice == 4 | choice == 5

fini_Agg_EP(1) = 1 - sum(Agg_popupT);
temp_Agg       = Agg_popupZ;

for ii=1:no_of_gen
    temp_Agg = Agg_Q' * temp_Agg;   
    
    fini_Agg_EP(length(fini_Agg_EP)+1) = sum (temp_Agg (opt_ind));
end

if  choice == 4 | choice == 5
    fini_low_EP(1) = 1 - sum(Agg_popupT);
    fini_up_EP(1)  = fini_low_EP(1);

    temp_low = Agg_popupZ;
    temp_up  = Agg_popupZ;

    for ii=1:no_of_gen
        temp_low = low_Q' * temp_low;
        temp_up  = up_Q'  * temp_up;
    
        fini_low_EP(length(fini_low_EP)+1) = sum (temp_low (opt_ind));
        fini_up_EP (length(fini_up_EP)+ 1) = sum (temp_up  (opt_ind));
    end
end % choice == 4 | choice == 5

disp (['time to compute aggregated finite population model = ', num2str(toc)]);

disp (' ');
disp ('aggregated finite population model finished');
disp (' ');

   
% limit the probability vectors to between 0 and 1

for x = 1:no_of_gen+1
    if Agg_P_first(x) > 1
       Agg_P_first(x) = 1;
    elseif Agg_P_first(x) < 0
       Agg_P_first(x) = 0;
    end
end

for x = 1:no_of_gen+1
    if fini_Agg_EP(x) > 1
       fini_Agg_EP(x) = 1;
    elseif fini_Agg_EP(x) < 0
       fini_Agg_EP(x) = 0;
    end
end

if  choice == 4 | choice == 5
    for x = 1:no_of_gen+1
        if up_P_first(x) > 1
            up_P_first(x) = 1;
        elseif up_P_first(x) < 0
            up_P_first(x) = 0;
        end
    end

    for x = 1:no_of_gen+1
        if low_P_first(x) > 1
            low_P_first(x) = 1;
        elseif low_P_first(x) < 0
            low_P_first(x) = 0;
        end
    end

    for x = 1:no_of_gen+1
        if up_P_Markov(x) > 1
            up_P_Markov(x) = 1;
        elseif up_P_Markov(x) < 0
            up_P_Markov(x) = 0;
        end
    end

    for x = 1:no_of_gen+1
        if low_P_Markov(x) > 1
            low_P_Markov(x) = 1;
        elseif low_P_Markov(x) < 0
            low_P_Markov(x) = 0;
        end
    end

    for x = 1:no_of_gen+1
        if fini_up_EP(x) > 1
            fini_up_EP(x) = 1;
        elseif fini_up_EP(x) < 0
            fini_up_EP(x) = 0;
        end
    end

    for x = 1:no_of_gen+1
        if fini_low_EP(x) > 1
            fini_low_EP(x) = 1;
        elseif fini_low_EP(x) < 0
            fini_low_EP(x) = 0;
        end
    end

end % choice == 4 | choice == 5

% DISPLAY - FINITE POPULATION MODEL
% ***********************************************************************************************

if  choice == 4 | choice == 5
    % display aggregation statistics
    disp ('Aggregation statistics');

    a = size(T, 1);
    b = no_Tlabel;

    x = num2str (a);
    disp ( ['number of states of T in full model     = ', x ] );
    x = num2str (b);
    disp ( ['number of states of T in reduced model  = ', x ] );
    x = num2str ( (a - b) * 100 /a  );
    disp ( ['Percentage of reduction            = ', x , ' %' ] );
    disp (' ');

end % choice == 4 | choice == 5

x = num2str (Agg_T_first);
disp (['approx. expected first hitting time     = ', x]);

if choice == 3

figure; hold on; 
title (['first passage probability']);
xlabel('generation'); ylabel('probability');

x = 0:no_of_gen;
if colour_plot
   plot(x, Agg_P_first,  'c-');
else
   plot(x, Agg_P_first,  'k-.'); 
end
 
legend('approx.', 0);

hold off;    
    
end % choice == 3


if choice == 4  | choice == 5

% display expected first hitting time statistics
if up_valid
   x = num2str (up_T_first);
   disp (['upper bound expected first hitting time = ', x]);
else
   disp ('the upper bound expected first hitting time cannot be estimated'); 
end
     
if choice == 5
   x = num2str (T_first);
   disp (['expected first hitting time             = ', x]);
end % choice == 5

x = num2str (low_T_first);
disp (['lower bound expected first hitting time = ', x]);


figure; hold on; 
title (['first passage probability']);
xlabel('generation'); ylabel('probability');

x = 0:no_of_gen;
if colour_plot
   plot(x, up_P_first,   'r-');
   if choice == 5
      plot(x, P_first,      'g-');
   end % choice == 5
   plot(x, Agg_P_first,  'c-');
   plot(x, low_P_first,  'b-');
else
   plot(x, up_P_first,   'k:');
   if choice == 5
      plot(x, P_first,      'k-');
   end % choice == 5
   plot(x, Agg_P_first,  'k-.');
   plot(x, low_P_first,  'k--');    
end
 
if choice == 5
   legend('upper bound', 'actual', 'approx.', 'lower bound', 0);
else
   legend('upper bound', 'approx.', 'lower bound', 0);
end % choice == 5

hold off;

if choice == 4 
    
if up_valid
    figure; hold on; 
    title (['first passage probability using Markov inequality']);
    xlabel('generation'); ylabel('probability');

    x = 0:no_of_gen;
    if colour_plot
        plot(x, up_P_Markov,  'r-'); 
    else
        plot(x, up_P_Markov,  'k:');       
    end

    legend('lower bound', 0);
    
    hold off;
else
    disp ('The lower bound of the first passage probability cannot be calculated using Markov inequality'); 
end

end % choice == 4

if choice == 5

figure; hold on; 
title (['first passage probability using Markov inequality']);
xlabel('generation'); ylabel('probability');

x = 0:no_of_gen;
if colour_plot
   if up_valid
      plot(x, up_P_Markov,  'r-'); 
   end
   plot(x, P_Markov,     'g-');  
else
   if up_valid
      plot(x, up_P_Markov,  'k:'); 
   end
   plot(x, P_Markov,     'k-');      
end

if up_valid
   legend('lower bound', 'actual', 0);
else
   legend(               'actual', 0);
end    
   
hold off;

end % choice == 5



end % choice == 4 | choice == 5

figure; hold on;
title (['expected  probability of optimal chromosome']);
xlabel ('generation'); ylabel ('probability');

x = 0:no_of_gen;
if  colour_plot
    plot(x, fini_Agg_EP,  'c-');
    plot(x, infi_EP,      'm-');
else
    plot(x, fini_Agg_EP,  'k-.');
    plot(x, infi_EP,      'k-');    
end

legend ('approx. (finite population model) ', 'actual (infinite population model)', 0);

hold off;


end % choice










