function rij = calculate_rij ( i, j, mu, chi_mu, gamma )

%  Function :   Calculate rij(0) in Vose-Liepins' infinite population model
%  
%  References:  This is an implementation of Eqn (3) J. Suzuki, "A Further result on the Markov chain model of genetic algorithms 
%               and its application to a simulated annealing-like strategy, IEEE T SMC Part B: Cybernetics
%               28(1), pp. 95-102, 1998.
%
%  Example i=9 j=5 mu=0.01 chi_mu=0.6 gamma=4
%          The result is supposed to be 
%          (0.6.^2+(1-0.6).^2)*0.01.^2*(1-0.01).^2+0.6*(1-0.6)*(0.01.^3*(1-0.01)+0.01*(1-0.01).^3)
%          calculated manually
%          The result is the same by running 
%          rij=calculate_rij(9,5,0.01,0.6,4)
%          rij =0.0024 
%              =(0.6.^2+(1-0.6).^2)*0.01.^2*(1-0.01).^2+0.6*(1-0.6)*(0.01.^3*(1-0.01)+0.0
%               1*(1-0.01).^3)
%  
%  Input Parameters :  i,j     integers with size between [0 2.^gamma-1]
%                      mu      bit mutation rate
%                      chi_mu  bit crossover rate
%
%  Output Parameters :  rij(0)
%


if nargin ~=5 
    warning('function calculate_rij has incorrect number of input');
    return;
end


% a is binary string of i, b is binary string of j
a=dec2bin(i,gamma);
b=dec2bin(j,gamma);

% c is a XOR b
% str2num(a(:)) is a must 
c=xor(str2num(a(:)),str2num(b(:)));
    
lengthcma=sum(c.*str2num(a(:)));   
lengthcmb=sum(c.*str2num(b(:)));
lengtha=sum(str2num(a(:)));
lengthb=sum(str2num(b(:)));
lengthc=sum(c);
    
rij=0;
    
for t=0:lengthc
    temp=0;
    for s=0:t
            
        if lengthcma < (t-s) continue;
        else
           one2zero=nchoosek(lengthcma ,t-s);   
        end
        if lengthcmb < s     continue; 
        else
           zero2one= nchoosek(lengthcmb ,s);   
        end
        temp=temp+ one2zero*zero2one*(mu.^(lengtha+2*s-t)*(1-mu).^(gamma-lengtha-2*s+t)+...
                                      mu.^(lengthb-2*s+t)*(1-mu).^(gamma-lengthb+2*s-t));
    end
    rij=rij+temp*chi_mu.^t*(1-chi_mu).^(lengthc-t);
end

rij = rij/2;