% -----  -----  -----  -----  -----  -----  -----  -----  -----  ----- %
% Source Code: Y.Lou and S.Y.Yuen, "On Constructing Alternative
%              Benchmark Suite for Evolutionary Algorithms", Swarm and
%              Evolutionary Computation, 44:287–292;
%              doi:10.1016/j.swevo.2018.04.005 (2019) 
% -----  -----  -----  -----  -----  -----  -----  -----  -----  ----- %
% Four Algorithms={'ABC'; 'CoDE'; 'SPSO2011'; 'NBiPOP-CMAES';}
%       1. ABC
%       2. CoDE
%       3. SPSO 2011
%       4. NBiPOP-CMA-ES
% -----  -----  -----  -----  -----  -----  -----  -----  -----  ----- %
% Use Gaussian Landscapes, and CEC 2013 Benchmark Suite
% -----  -----  -----  -----  -----  -----  -----  -----  -----  ----- %
% Copyright (c) 2019 Yang Lou and Shiu Yin Yuen. 
% All rights reserved.
% 
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions
% are met: 
% 
% 1. Redistributions of source code must retain the above copyright
%    notice, this list of conditions and the following disclaimer. 
% 2. Redistributions in binary form must reproduce the above copyright
%    notice, this list of conditions and the following disclaimer in
%    the documentation and/or other materials provided with the 
%    distribution.
% 
% This software is provided by the copyright holders and contributors
% "as is" and any express or implied warranties, including, but not 
% limited to, the implied warranties of merchantability and fitness
% for a particular purpose are disclaimed. In no event shall the
% copyright owner or contributors be liable for any direct, indirect,
% incidental, special, exemplary, or consequential damages (including,
% but not limited to, procurement of substitute goods or services;
% loss of use, data, or profits; or business interruption) however
% caused and on any theory of liability, whether in contract, strict
% liability, or tort (including negligence or otherwise) arising in
% any way out of the use of this software, even if advised of the
% possibility of such damage.
% -----  -----  -----  -----  -----  -----  -----  -----  -----  ----- %
% Updated: 25-09-2020
% -----  -----  -----  -----  -----  -----  -----  -----  -----  ----- %

clearvars; clc;
global arXiv;
global fMat;
global fit;

disp('Settings for component EAs ...')
disp('Generate Uniquely-Easy Problem for which EA?')
TM=input('(1) ABC, (2) CoDE, (3) SPSO-2011, (4) NBiPOPCMAES: ');
nGauss=input('Number of Modals (Local Optima), (e.g.,5): ');
Dim=input('Problem Dimension, (at minimum 2, e.g.,10): ');
Rept=input('Number of Repeated Runs, (e.g.,20): ');

% ----- >> Set Here -----  -----  ----- %
BenchMarks={'cec2013'; 'glg06';};
BM=2;  %% BenckMark - ID (1-CEC || 2-GLG)
Algorithms={'abc'; 'code'; 'spso'; 'nbicma';};
TestMode={'abcs'; 'cods'; 'psos'; 'cmas'; };
NEA=length(Algorithms);
ProbSet=BenchMarks{BM};
fmode=TestMode{TM};
disp(['BenchMark(',ProbSet,') || TestMode(', fmode, ') ... ']);
addpath(genpath('algorithms'))

switch ProbSet
    case 'glg06'
        % ----- >> Gaussian Landscape Paras ----- %
        set=struct( ...	%% Gaussian Benchmark Setting
                    'nGauss',	nGauss,	...
                    'd',        Dim,	...
                    'gbf',      100,	...
                    'ub',       100,	...
                    'lb',       -100	...
                    );
        set.sigb.ub=(set.ub-set.lb)/1;  %% Note "set.ub" & "set.sigb.ub" Different
        set.sigb.lb=1E-9;
        set.maxfes=min(1E3*set.d*set.nGauss,1E5);
    case 'cec2013'
        % ----- >> Gaussian Landscape Paras ----- %
        cecProbID=1:28;  %% 1~28
        set=struct( ...	%% CEC Benchmarks Setting
                    'd',    Dim,	...
                    'ub',	100,	...
                    'lb',	-100,	...
                    'fn',   0       ...  %% Benchmark Function ID CEC2013
                    );
        set.maxfes=1E4*set.d;
end

%% -----  -----  -----| Generating New Instances |-----  -----  ----- %%
if strcmp(ProbSet,'glg06')
    % Use Differential Evolution to Evolve Instance in Gaussian Landscapes
    D=set.nGauss*(set.d+2);
        % (nGauss,1)	>> Sigma
        % (nGauss,d)	>> Squeeze
        % (nGauss-1,1)	>> Ratio
        % Plus a Random Seed 'rs', (nGauss,d+2) in Total
    F=0.85;
    CR=0.5;       %% Para Setting: http://www1.icsi.berkeley.edu/~storn/code.html
    psDE=20;
    MAXFE=1000;
    % -----  -----  -----  -----  ----- 
%     if (1)
%         Rept=5;
%         psDE=4;
%         MAXFE=20;
%     end
    % -----  -----  -----  -----  ----- 
    MAXG=MAXFE/psDE;
    disp('Using Gaussian Landscape Generator to Evolve New Benchmark Instances: ')
    disp(['>> nGauss(',int2str(set.nGauss),');  Dim(',int2str(set.d),');  Repeat(',int2str(Rept),') ...'])
    disp(['   PopSize(',int2str(psDE),');  MaxFEs_DE(',int2str(MAXFE),') ...'])
    if psDE<3
        error('Population Size of Differential Evolution Should be Greater Than 3, for Mutation Purpose.')
    end
    % ----- Initialization ----- %
    % Note Size(x_DE)	=D-by-psDE;
    %      Size(x_Gauss)=ps-by-d;
    x=zeros(D,psDE);	%% Configuration (Paras) of Benchmark Instances
    f=cell(psDE,1);	%% Fitness of Benchmark Instances
    g=1;  disp(['Initializing (Gen.',int2str(g),') ...'])
    fmat=cell(psDE,1);  rs=zeros(psDE,1);  fbias=zeros(psDE,1);
    for idx=1:psDE  %% Initialize All Solutions, Eight Pops. %
        % ----- Config-to-X ----- %
        sig=set.sigb.lb+rand(set.nGauss,1)*(set.sigb.ub - set.sigb.lb);	% (nGauss,1)	>> Sigma
        sqz=reshape(rand(set.nGauss,set.d),set.nGauss*set.d,1);           % (nGauss,d)	>> Squeeze
        rat=rand(set.nGauss-1,1);                                         % (nGauss-1,1)	>> Ratio
        rs=randi(2^53-1);  % IMAX Less Than 2^53
        % ----- Config-to-X ----- %
        x(1:D,idx)=[sig;sqz;rat;rs];
        [f{idx,1},fmat{idx},fbias(idx)]=feval_prob(ProbSet,set,x(:,idx),Rept,fmode);
    end
    % ----- Update Current Best ----- %
    [fmin,xmin,dx]=find_best(f,x);
    fmin_mat=fmat{dx};    %% Fit of EAs Under Best Configuration
    % ----- arXiv Initialization & Insertion ----- %
    [arXiv,nNode,fMat,fit,nFmat]=arXiv_init(D,MAXFE);  %% Was: [arXiv,nNode,fMat,nFmat]=arXiv_init(D,MAXFE);
    % Note "fMat=cell(cell(MAXFE,2)" and "fit=cell(mfe,2)". %
    [nNode,nFmat]=arXiv_in(x,f,nNode,fmat,nFmat,fbias);
    fmin
    while (g<MAXG)&&(fmin.f>1E-5)  %% (fmin.f>1E-10)
        g=g+1;
        disp(['Running Gen.',int2str(g),' ...'])
        % ----- Generate Offspring ----- %
        u=zeros(D,psDE);
        for idx=1:psDE
            i=randperm(psDE);            
            i=i(1:3);  %% psDE > 3
            v=x(:,i(1))+F*(x(:,i(2))-x(:,i(3)));
            cr=rand(D,1)>CR;
            u(:,idx)=cr.*v + (~cr).*x(:,idx);
        end
        u=bound_check(u,set.nGauss,set.sigb);  %% Sigb is Bounds of Sigma
        % ----- arXiv Revisit Check ----- %
        u=arXiv_check(u,set.lb,set.ub);
        fmat=cell(psDE,1);  fu=cell(psDE,1);
        for idx=1:psDE  %% Evaluate One after Another
            %%[fu(idx,1),fmat{idx},fbias(idx)]=feval_prob(ProbSet,set,u(:,idx),Rept,fmode);
            [fu{idx},fmat{idx},fbias(idx)]=feval_prob(ProbSet,set,u(:,idx),Rept,fmode);
            % ----- Tournament Selection ----- %
            [f{idx},pdx]=tour_sele(fu{idx},f{idx}); 
            if pdx==1  %% Meaning 'fu{idx}' is Better
                x(:,idx)=u(:,idx);
            end  %% {x,f} is the next-generation-population
        end
        [nNode,nFmat]=arXiv_in(u,fu,nNode,fmat,nFmat,fbias);  %% U -> arXiv, No Matter Better or Not
        % ----- Update Current Best ----- %
        [fmin,xmin,dx]=find_best(f,x);
        fmin_mat=fmat{dx};    %% Fit of EAs Under Best Configuration
        clear fmat
        fmin
    end
    % save all information
    fname=['all_N',int2str(set.nGauss),'D',int2str(set.d), ...
             'UE',Algorithms{TM},'.mat'];
    save(fname)
    % save only the required problem instance
    fname=['bestInst_N',int2str(set.nGauss),'D',int2str(set.d), ...
             'UE',Algorithms{TM},'.mat'];
    save(fname,'xmin')
    
% -----  -----  ----- Testing -----  -----  ----- %%
elseif strcmp(ProbSet,'cec2013')
    clear arXiv fMat fit TestMode fmode
    % Testing on CEC 2013, Not to Evolve any New Instances
    disp('Testing on CEC 2013 Benchmarks ... ')
    disp(['>> Dim(',int2str(set.d),'); Rept(',int2str(Rept),') ...'])
    NPB=cecProbID(end);  %% #Problems
    fmat=cell(NPB,1);
    for idx=cecProbID
        disp(['Benchmark CEC 2013 - f',int2str(idx), ' ...']);
        set.fn=idx;
        [~,fmat{idx,1},~]=feval_prob(ProbSet,set,[],Rept,[]);
    end
    fname=['v6_cec13_d',int2str(set.d),'.mat'];
    save(fname)
    
end

