package onePositionInheritanceABC;

import java.lang.Math;
import java.util.Random;

public class FitnessFunction 
{
	/**
	 * define the fitness functions and compute the fitness value
	 */
	public FitnessFunction(int xdim)
	{
		this.xdim = xdim;
	}
	
	/**
	 * compute fitness (function) value along with the funID and x
	 * @param funID fitness function identity
	 * @param x independent values of fitness function funID
	 * @param isrotate 
	 * @param om rotation matrix
	 * @return 
	 */
	public void computeFitness(int funID, double[] xCopy, int isrotate, double[][] om)
	{
		fitness = 0.0;
		
		if(xCopy.length != xdim)
		{
			throw new IllegalArgumentException("Unexpected x length in FitnessFunction.java!");
		}
		double[] x = new double[xdim];
		// initialize x to 0
		for(int ix = 0; ix < xdim; ix++)
		{
			x[ix] = 0.0;
		}
		if(isrotate == 0)
		{
			// set array x to original individual
			for(int ix = 0; ix < xdim; ix++)
			{
				x[ix] = xCopy[ix];
			}
		}
		else if(isrotate == 1)
		{
			// compute rotated individual
			for(int ix = 0; ix < xdim; ix++)
			{
				for(int iRow = 0; iRow < xdim; iRow++)
				{
					x[ix] = x[ix] + xCopy[iRow] * om[iRow][ix];
				}
			}
		}
		else
		{
			throw new IllegalArgumentException("Please pass a fitness computation method!");
		}
		
		// sphere function
		if(funID == 1)
		{
			for(int ix = 0; ix < x.length; ix++)
			{
				fitness = fitness + x[ix] * x[ix];
			}
		}
		
		// schwefel problem 2.22
		if(funID == 2)
		{
			fitness = 0.0;
			
			double tmp1 = 0.0;
			double tmp2 = 1.0;
			for(int ixdim = 0; ixdim < xdim; ixdim++)
			{
				tmp1 = tmp1 + Math.abs(x[ixdim]);
				tmp2 = tmp2 * Math.abs(x[ixdim]);
			}
			fitness = tmp1 + tmp2;
		}
		
		// schwefel problem 1.2
		if(funID == 3)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			
			for(int ixdim = 0; ixdim < xdim; ixdim++)
			{
				double tmp2 = 0.0;
				for(int ix = 0; ix <= ixdim; ix++)
				{
					tmp2 = tmp2 + x[ix]; 
				}
				tmp1 = tmp1 + tmp2 * tmp2;
			}
			fitness = tmp1;
		}
		
		// schwefel problem 2.21
		if(funID == 4)
		{
			fitness = Math.abs(x[0]);
						
			for(int ixdim = 1; ixdim < xdim; ixdim++)
			{
				double tmp1 = Math.abs(x[ixdim]);
				if(tmp1 > fitness)
				{
					fitness = tmp1;
				}
			}
		}
		
		// generalized rosenbrock function
		if(funID == 5)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			for(int ix = 0; ix < (xdim - 1); ix++)
			{
				tmp1 = tmp1 + (100 * (x[ix+1] - x[ix]*x[ix]) * (x[ix+1] - x[ix]*x[ix]) +
				(x[ix] - 1) * (x[ix] - 1));
			}
			fitness = tmp1;
			
		}
		
		// quaric function
		if(funID == 6)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			Random generator = new Random();
			
			for(int ix = 0; ix < xdim; ix++)
			{
				tmp1 = tmp1 + ix * Math.pow(x[ix], 4);
			}
			fitness = tmp1 + generator.nextDouble();
		}
		
		// generalized rastrigin function
		if(funID == 7)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				tmp1 = tmp1 + (x[ix] * x[ix] - 10 * Math.cos(2*Math.PI*x[ix]) + 10);
			}
			fitness = tmp1;
		}
		
		// generalized griewank function
		if(funID == 8)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			double tmp2 = 1.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				tmp1 = tmp1 + x[ix] * x[ix];
				tmp2 = tmp2 * Math.cos(x[ix] / Math.sqrt(ix+1));
			}
			fitness = tmp1 / 4000.0 - tmp2 + 1.0;
		}
		
		// generalized schwefel problem 2.26
		if(funID == 9)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				tmp1 = tmp1 + x[ix] * Math.sin(Math.sqrt(Math.abs(x[ix])));
			}
			fitness = -tmp1;
		}
		
		// ackley function
		if(funID == 10)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			double tmp2 = 0.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				tmp1 = tmp1 + x[ix] * x[ix];
				tmp2 = tmp2 + Math.cos(2.0 * Math.PI * x[ix]);
			}
			
			fitness = -20.0 * Math.exp(-0.2 * Math.sqrt(tmp1 / xdim)) - 
			Math.exp(tmp2 / xdim) + 20.0 + Math.exp(1.0);
		}
		
		// weierstrass function
		if(funID == 11)
		{
			fitness = 0.0;
			double a = 0.5;
			double b = 3.0;
			
			double tmp1 = 0.0;
			double tmp2 = 0.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				for(int ik = 0; ik <= 20; ik++)
				{
					tmp1 = tmp1 + Math.pow(a, ik) * Math.cos(2.0*Math.PI*Math.pow(b, ik) * (x[ix] + 0.5));
				}
			}
			for(int ik = 0; ik < 20; ik++)
			{
				tmp2 = tmp2 + Math.pow(a, ik) * Math.cos(Math.PI * Math.pow(b, ik));
			}
			fitness = tmp1 - xdim * tmp2;
			if(fitness < 0.0)
			{
				fitness = 0.0;
			}
		}
		
		// levy function 1
		if(funID == 12)
		{
			fitness = 0.0;
			double[] y = new double[xdim];
			for(int ix = 0; ix < xdim; ix++)
			{
				y[ix] = 1.0 + (x[ix] - 1.0) / 4.0;
			}
			double tmp1 = 0.0;
			for(int iy = 0; iy < (xdim - 1); iy++)
			{
				tmp1 = tmp1 + (y[iy] - 1.0) * (y[iy] - 1.0) * 
				(1.0 + 10.0 * Math.sin(Math.PI * y[iy+1]) * Math.sin(Math.PI * y[iy+1]));
				
			}
			fitness = Math.sin(Math.PI * y[0])*Math.sin(Math.PI * y[0]) + tmp1 + 
			(y[xdim-1] - 1.0) * (y[xdim-1] - 1.0) * 
			(1.0 + 10.0 * Math.sin(2.0*Math.PI * y[xdim-1]) * Math.sin(2.0*Math.PI * y[xdim-1]));
		}
		
		// zakharov function
		if(funID == 13)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			double tmp2 = 0.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				tmp1 = tmp1 + x[ix] * x[ix];
				tmp2 = tmp2 + 0.5 * (ix + 1.0) * x[ix];
			}
			fitness = tmp1 + tmp2 * tmp2 + Math.pow(tmp2, 4);
		}
		
		// alpine function
		if(funID == 14)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				tmp1 = tmp1 + Math.abs(x[ix] * Math.sin(x[ix]) + 0.1 * x[ix]);
			}
			fitness = tmp1;
		}
		
		// pathological function
		if(funID == 15)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			double tmp2 = 0.0;
			for(int ix = 0; ix < (xdim - 1); ix++)
			{
				tmp1 = Math.sin(Math.sqrt(100.0*x[ix]*x[ix] + x[ix+1]*x[ix+1]));
				tmp1 = tmp1 * tmp1 - 0.5;
				tmp2 = (x[ix]*x[ix] - 2*x[ix]*x[ix+1] + x[ix+1]*x[ix+1]);
				tmp2 = 1 + 0.001 * tmp2*tmp2;
				
				fitness = fitness + (0.5 + tmp1 / tmp2);
			}
		}
		
		// inverted cosine wave function
		if(funID == 16)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			double tmp2 = 0.0;
			for(int ix = 0; ix < (xdim-1); ix++)
			{
				tmp1 = (x[ix]*x[ix] + x[ix+1]*x[ix+1] + 0.5*x[ix]*x[ix+1]);				
				tmp2 = Math.exp(-tmp1 / 8.0) * Math.cos(4.0*Math.sqrt(tmp1));
				fitness = fitness + tmp2;
			}
			fitness = -fitness;
		}
		
		// inverted cosine mixture problem
		if(funID == 17)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			double tmp2 = 0.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				tmp1 = tmp1 + Math.cos(5.0*Math.PI*x[ix]);
				tmp2 = tmp2 + x[ix]*x[ix];
			}
			fitness = 0.1*xdim - (0.1 * tmp1 - tmp2);
		}
		
		// epistatic michalewicz problem
		if(funID == 18)
		{
			fitness = 0.0;
			double[] y = new double[xdim];
			double tmp1 = 0.0;
			for(int ixdim = 0; ixdim < (xdim-1); ixdim++)
			{
				if(ixdim % 2 == 0)
				{
					y[ixdim] = x[ixdim]*Math.cos(Math.PI/6.0) - x[ixdim+1]*Math.sin(Math.PI/6.0);
				}
				else if(ixdim % 2 == 1)
				{
					y[ixdim] = x[ixdim]*Math.sin(Math.PI/6.0) + x[ixdim+1]*Math.cos(Math.PI/6.0);
				}
			}
			y[xdim-1] = x[xdim-1];
			
			for(int iy = 0; iy < xdim; iy++)
			{
				tmp1 = tmp1 + Math.sin(y[iy]*y[iy]) * Math.pow(Math.sin((iy+1)*y[iy]*y[iy]/Math.PI), 20);
			}
			fitness = -tmp1;
		}
		
		// levy and montalvo 2 problem
		if(funID == 19)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			for(int ix = 0; ix < (xdim-1); ix++)
			{
				tmp1 = tmp1 + (x[ix] - 1)*(x[ix] - 1) * (1.0 + 10.0*Math.sin(3.0*Math.PI*x[ix+1])*Math.sin(3.0*Math.PI*x[ix+1]));
			}
			fitness = 0.1*(Math.sin(3.0*Math.PI*x[0])*Math.sin(3.0*Math.PI*x[0]) + 
					tmp1 + (x[xdim-1] - 1)*(x[xdim-1] - 1)*(1.0 + 10.0*Math.sin(2.0*Math.PI*x[xdim-1])*Math.sin(2.0*Math.PI*x[xdim-1])));
			
		}
		
		// periodic problem
		if(funID == 20)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			double tmp2 = 1.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				tmp1 = tmp1 + (Math.sin(x[ix]) * Math.sin(x[ix]));
				tmp2 = tmp2 * Math.exp(-x[ix]*x[ix]);
			}
			fitness = 1 + tmp1 - 0.1 * tmp2;
		}
		
		// salomon problem
		if(funID == 21)
		{
			fitness = 0.0;
			double tmp1 = 0.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				tmp1 = tmp1 + x[ix] * x[ix];
			}
			fitness = 1 - Math.cos(2.0*Math.PI*Math.sqrt(tmp1)) + 0.1*Math.sqrt(tmp1);
		}
		
		// shubert problem
		if(funID == 22)
		{
			fitness = 1.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				double tmp1 = 0.0;
				for(int jcol = 1; jcol <= 5; jcol++)
				{
					tmp1 = tmp1 + jcol*Math.cos((jcol+1)*x[ix] + jcol);
				}
				fitness = fitness * tmp1;
			}
		}
		
		// sinusoidal problem
		if(funID == 23)
		{
			fitness = 0.0;
			double A = 2.5;
			double B = 5.0;
			double z = 30;
			double tmp1 = 1.0;
			double tmp2 = 1.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				tmp1 = tmp1 * Math.sin((x[ix]-z)*Math.PI/180.0);
				tmp2 = tmp2 * Math.sin(B*(x[ix]-z)*Math.PI/180.0);
			}
			fitness = -(A * tmp1 + tmp2);
		}
		
		// michalewicz function
		if(funID == 24)
		{
			fitness = 0.0;
			int m = 10;
			double tmp1 = 0.0;
			for(int ix = 0; ix < xdim; ix++)
			{
				tmp1 = tmp1 + Math.sin(x[ix]*x[ix]) * Math.pow(Math.sin((ix+1)*x[ix]*x[ix]/Math.PI), 2*m);
			}
			fitness = -tmp1;
		}
		
		// whitely function
		if(funID == 25)
		{
			fitness = 0.0;
			double yij;
			for(int irow = 0; irow < xdim; irow++)
			{
				double tmp1 = (1.0 - x[irow]) * (1.0 - x[irow]);
				for(int jcol = 0; jcol < xdim; jcol++)
				{
					yij = 100.0 * (x[jcol] - x[irow]*x[irow]) * (x[jcol] - x[irow]*x[irow]) + tmp1;
					fitness = fitness + (yij/4000.0 - Math.cos(yij) + 1.0);
				}
			}
		}
	}// computeFitness
	
	/**
	 * @return function (fitness) value of solution x
	 */
	public double getFitness()
	{
		return fitness;
	}

	private int xdim; // problem (function) dimension 
	private double fitness; // function value of a solution
}